<?php
declare(strict_types=1);

final class ChatController {
  public static function targets(string $incidentId): void {
    $u = require_role(['admin','el','gf']);
    $pdo = db();

    if ($u['role'] !== 'admin') {
      $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
      $chk->execute([':iid'=>$incidentId, ':oid'=>$u['org_id']]);
      if (!$chk->fetchColumn()) api_error(403, 'not a participant');
    }

    // Orgs in incident
    $orgStmt = $pdo->prepare('SELECT o.id, o.name, o.short_name, o.color
      FROM incident_org io
      JOIN org o ON o.id=io.org_id
      WHERE io.incident_id=:iid
      ORDER BY o.name');
    $orgStmt->execute([':iid'=>$incidentId]);
    $orgs = $orgStmt->fetchAll();

    // Users in those orgs
    $userStmt = $pdo->prepare('SELECT u.id, u.username, u.display_name, u.role, u.org_id, o.short_name AS org_short
      FROM app_user u
      JOIN org o ON o.id=u.org_id
      WHERE u.is_active=true
        AND EXISTS (SELECT 1 FROM incident_org io WHERE io.incident_id=:iid AND io.org_id=u.org_id)
      ORDER BY o.short_name, u.display_name');
    $userStmt->execute([':iid'=>$incidentId]);
    $users = $userStmt->fetchAll();

    api_json([
      'ok'=>true,
      'orgs'=>$orgs,
      'users'=>$users,
      'special'=>[
        ['key'=>'all','label'=>'Alle'],
        ['key'=>'gfs','label'=>'GFs/EL'],
      ],
    ]);
  }

  public static function list(string $incidentId): void {
    $u = require_login();
    $pdo = db();
    if ($u['role'] !== 'admin') {
      $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
      $chk->execute([':iid'=>$incidentId, ':oid'=>$u['org_id']]);
      if (!$chk->fetchColumn()) api_error(403, 'not a participant');
    }

    $since = (string)($_GET['since'] ?? '');
    $limit = (int)($_GET['limit'] ?? 200);
    if ($limit < 1) $limit = 200;
    if ($limit > 500) $limit = 500;

    $sql = 'SELECT m.id, m.author_user_id, m.incident_id, m.scope, m.org_id, m.to_org_id, m.to_user_id, m.message,
        m.mentions_json, m.created_at,
        u.display_name AS author_name, u.username AS author_username, o.short_name AS author_org
      FROM message m
      JOIN app_user u ON u.id=m.author_user_id
      JOIN org o ON o.id=u.org_id
      WHERE m.incident_id=:iid';
    $params = [':iid'=>$incidentId];
    if ($since !== '') {
      $sql .= ' AND m.created_at > :since';
      $params[':since'] = $since;
    }
    $sql .= ' ORDER BY m.created_at DESC LIMIT ' . $limit;

    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $rows = array_reverse($stmt->fetchAll());
    foreach ($rows as &$r) {
      $r['mentions'] = json_decode($r['mentions_json'] ?? '[]', true);
      unset($r['mentions_json']);
    }

    // Scope filtering in PHP (simple)
    $out = [];
    foreach ($rows as $r) {
      if ($r['scope']==='public') { $out[] = $r; continue; }
      if ($r['scope']==='org' && $r['to_org_id']===$u['org_id']) { $out[] = $r; continue; }
      if ($r['scope']==='org' && $r['org_id']===$u['org_id']) { $out[] = $r; continue; }
      if ($r['scope']==='private' && (string)$r['to_user_id']===(string)$u['id']) { $out[] = $r; continue; }
      if ($r['scope']==='private' && (string)$r['author_user_id']===(string)$u['id']) { $out[] = $r; continue; }
      if ($r['scope']==='gfs' && in_array($u['role'], ['gf','el','admin'], true)) { $out[]=$r; continue; }
      if ($r['scope']==='all') { $out[]=$r; continue; }
    }

    api_json(['ok'=>true, 'messages'=>$out]);
  }

  public static function send(string $incidentId): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    $scope = (string)($in['scope'] ?? 'public'); // public|org|gfs|all|private
    $toOrgId = (string)($in['to_org_id'] ?? '');
    $toUserId = (string)($in['to_user_id'] ?? '');
    $msg = trim((string)($in['message'] ?? ''));
    $mentions = $in['mentions'] ?? [];
    if (!is_array($mentions)) $mentions = [];
    if ($msg==='') api_error(400, 'message required');

    // Validate scope
    $allowed = ['public','org','gfs','all','private'];
    if (!in_array($scope, $allowed, true)) api_error(400, 'invalid scope');

    // Default org scope: send to own org
    if ($scope==='org' && $toOrgId==='') $toOrgId = $u['org_id'];

    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO message(id, incident_id, author_user_id, scope, org_id, to_org_id, to_user_id, message, mentions_json)
      VALUES (gen_random_uuid(), :iid, :uid, :sc, :oid, NULLIF(:to_oid,\'\')::uuid, NULLIF(:to_uid,\'\')::int, :m, :men::jsonb)
      RETURNING id');
    $stmt->execute([
      ':iid'=>$incidentId, ':uid'=>$u['id'], ':sc'=>$scope, ':oid'=>$u['org_id'],
      ':to_oid'=>$toOrgId, ':to_uid'=>$toUserId,
      ':m'=>$msg,
      ':men'=>json_encode($mentions, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
    ]);
    $id = (string)$stmt->fetchColumn();

    // Emit events depending on scope
    if ($scope==='public' || $scope==='all' || $scope==='gfs') {
      EventsController::emitIncident($incidentId, null, 'all', 'chat.updated', ['incident_id'=>$incidentId]);
    } elseif ($scope==='org') {
      EventsController::emitIncident($incidentId, $toOrgId, 'org', 'chat.updated', ['incident_id'=>$incidentId]);
      if ($toOrgId !== $u['org_id']) {
        EventsController::emitIncident($incidentId, $u['org_id'], 'org', 'chat.updated', ['incident_id'=>$incidentId]);
      }
    } elseif ($scope==='private') {
      // notify sender org and recipient org
      EventsController::emitIncident($incidentId, $u['org_id'], 'org', 'chat.updated', ['incident_id'=>$incidentId]);
      if ($toUserId !== '') {
        $q = $pdo->prepare('SELECT org_id FROM app_user WHERE id=:id');
        $q->execute([':id'=>(int)$toUserId]);
        $toUserOrg = (string)($q->fetchColumn() ?: '');
        if ($toUserOrg !== '' && $toUserOrg !== $u['org_id']) {
          EventsController::emitIncident($incidentId, $toUserOrg, 'org', 'chat.updated', ['incident_id'=>$incidentId]);
        }
      }
    }

    api_json(['ok'=>true, 'id'=>$id], 201);
  }

  public static function pins(string $incidentId): void {
    $u = require_login();
    $pdo = db();
    if ($u['role'] !== 'admin') {
      $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
      $chk->execute([':iid'=>$incidentId, ':oid'=>$u['org_id']]);
      if (!$chk->fetchColumn()) api_error(403, 'not a participant');
    }

    $stmt = $pdo->prepare('SELECT p.id AS pin_id, p.message_id, p.pinned_at, p.pinned_by_user_id,
        m.author_user_id, m.incident_id, m.scope, m.org_id, m.to_org_id, m.to_user_id, m.message, m.mentions_json, m.created_at,
        u.display_name AS author_name, u.username AS author_username, o.short_name AS author_org
      FROM message_pin p
      JOIN message m ON m.id=p.message_id
      JOIN app_user u ON u.id=m.author_user_id
      JOIN org o ON o.id=u.org_id
      WHERE p.incident_id=:iid
      ORDER BY p.sort_order ASC, p.pinned_at DESC');
    $stmt->execute([':iid'=>$incidentId]);
    $rows = $stmt->fetchAll() ?: [];
    foreach ($rows as &$r) {
      $r['mentions'] = json_decode($r['mentions_json'] ?? '[]', true);
      unset($r['mentions_json']);
    }

    // Apply same scope filtering as list()
    $out = [];
    foreach ($rows as $r) {
      if ($r['scope']==='public') { $out[] = $r; continue; }
      if ($r['scope']==='org' && $r['to_org_id']===$u['org_id']) { $out[] = $r; continue; }
      if ($r['scope']==='org' && $r['org_id']===$u['org_id']) { $out[] = $r; continue; }
      if ($r['scope']==='private' && (string)$r['to_user_id']===(string)$u['id']) { $out[] = $r; continue; }
      if ($r['scope']==='private' && (string)$r['author_user_id']===(string)$u['id']) { $out[] = $r; continue; }
      if ($r['scope']==='gfs' && in_array($u['role'], ['gf','el','admin'], true)) { $out[]=$r; continue; }
      if ($r['scope']==='all') { $out[]=$r; continue; }
    }

    api_json(['ok'=>true, 'pins'=>$out]);
  }

  public static function pin(string $incidentId, string $messageId): void {
    $u = require_role(['admin','el','gf']);
    $pdo = db();
    if ($u['role'] !== 'admin') {
      $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
      $chk->execute([':iid'=>$incidentId, ':oid'=>$u['org_id']]);
      if (!$chk->fetchColumn()) api_error(403, 'not a participant');
    }

    $in = json_input();
    $pinned = $in['pinned'] ?? true;
    $want = true;
    if (is_bool($pinned)) $want = $pinned;
    elseif (is_numeric($pinned)) $want = ((int)$pinned) === 1;
    elseif (is_string($pinned)) {
      $v = strtolower(trim($pinned));
      $want = in_array($v, ['1','true','yes','y','on'], true);
    }

    // Ensure message exists and belongs to incident
    $ms = $pdo->prepare('SELECT 1 FROM message WHERE id=:mid AND incident_id=:iid');
    $ms->execute([':mid'=>$messageId, ':iid'=>$incidentId]);
    if (!$ms->fetchColumn()) api_error(404, 'message not found');

    if ($want) {
      $stmt = $pdo->prepare('INSERT INTO message_pin(id, incident_id, message_id, pinned_by_user_id)
        VALUES (gen_random_uuid(), :iid, :mid, :uid)
        ON CONFLICT (incident_id, message_id) DO UPDATE SET pinned_by_user_id=EXCLUDED.pinned_by_user_id, pinned_at=now()');
      $stmt->execute([':iid'=>$incidentId, ':mid'=>$messageId, ':uid'=>$u['id']]);
    } else {
      $stmt = $pdo->prepare('DELETE FROM message_pin WHERE incident_id=:iid AND message_id=:mid');
      $stmt->execute([':iid'=>$incidentId, ':mid'=>$messageId]);
    }

    EventsController::emitIncident($incidentId, null, 'all', 'chat.pins.updated', ['incident_id'=>$incidentId]);
    api_json(['ok'=>true]);
  }
}
